%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Function name: apply_cal_corr
%
% Scope:    apply calibration correction to delta-values
%
% Input:    data (structure),
%           bindata (structure),
%           binref (structure),
%           calibration corr index (1-point vs 2-point) (double),
%           cal gas 1 name (string),
%           true values cal gas 1 (double),
%           cal gas 2 name (string),
%           true values cal gas 2 (double),
%           calibration corr index (isotopologue) (double),
%           list of gas names (cell),
%           list of gas type "flags" (i.e., MPV positions) (double),
%           index plot requested (logical)
%
% Output:   corrected "bindata" and "binref" structures,
%           overview figures (if requested)
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [ref, binref, bindata] = apply_cal_corr(data, bindata, binref, calib_idx, true_gas1, true_values1, true_gas2, true_values2, calib_idx_delta, gasnames, gastypeflag, plot_requested)
    
    tic;fprintf('Running function apply_cal_corr...\n');
    
    ind_gas1=find(ismember(gasnames,true_gas1));
    ind_gas2=find(ismember(gasnames,true_gas2));
        
    target_idx=[];
    if calib_idx_delta(1,1)==1
        target_idx(1,width(target_idx)+1)=4;
    end
    
    if calib_idx_delta(1,2)==1
        target_idx(1,width(target_idx)+1)=5;
    end
    
    if calib_idx_delta(1,3)==1
        target_idx(1,width(target_idx)+1)=6;
    end
    
    if width(target_idx)== 1 || width(target_idx)== 2
        if target_idx(1,1) == 5
           true_values1(1,1)=true_values1(1,2); 
           true_values2(1,1)=true_values2(1,2); 
        end
        if target_idx(1,1) == 6
           true_values1(1,1)=true_values1(1,3); 
           true_values2(1,1)=true_values2(1,3); 
        end
    end
        
    if width(target_idx)== 2
        if target_idx(1,2) == 6
           true_values1(1,2)=true_values1(1,3); 
           true_values2(1,2)=true_values2(1,3); 
        end
    end
    
    %% one-point calibration
    
    if calib_idx==1
        
        % plot data before correction
        if plot_requested    
            plot_1point_cal_corr(data,bindata,binref,gasnames,target_idx,'Before calibration correction');
        end
        
        % apply correction
        for z=1:width(target_idx)
            for i=1:width(gastypeflag)
                   bindata.(gasnames{1,i})(:,target_idx(1,z))=bindata.(gasnames{1,i})(:,target_idx(1,z))+(true_values1(1,z)-binref.(gasnames{1,ind_gas1})(1,target_idx(1,z)));
            end
            bindata.all(:,target_idx(1,z))=bindata.all(:,target_idx(1,z))+(true_values1(1,z)-binref.(gasnames{1,ind_gas1})(1,target_idx(1,z)));    
        end
        
        % recalculate ref conditions
        [ref, binref, bindata] = calc_ref_conditions(data,bindata,gasnames,gastypeflag);
        
        % plot data after correction
        if plot_requested 
            plot_1point_cal_corr(data,bindata,binref,gasnames,target_idx,'After calibration correction');
        end
        
    end
    
    %% two-point calibration
    
    if calib_idx==2 %two-point calibration

        % calculate linear fit coefficients true vs measured
        sss=NaN(width(target_idx),2);
        sssplot=NaN(width(target_idx),2);
        for i=1:width(target_idx)
            true=[true_values1(1,i) true_values2(1,i)];
            measured=[binref.(gasnames{1,ind_gas1})(1,target_idx(1,i)) binref.(gasnames{1,ind_gas2})(1,target_idx(1,i))];
            [sss(i,:),~]=polyfit(measured,true,1);
            [sssplot(i,:),~]=polyfit(true,measured,1);
        end
        
        %store slope and uncorrected values for uncertainty propagation
        bindata.calcorr.slope=NaN(3,2);
        for i=1:width(target_idx)
            if target_idx(1,i)==4
                bindata.calcorr.slope(1,:) = sss(i,:);
            end
            if target_idx(1,i)==5
                bindata.calcorr.slope(2,:) = sss(i,:);
            end
            if target_idx(1,i)==6
                bindata.calcorr.slope(3,:) = sss(i,:);
            end
        end
        
        bindata.calcorr.all=bindata.all;     
        for i=1:width(gastypeflag)
               bindata.calcorr.(gasnames{1,i})=bindata.(gasnames{1,i});     
               bindata.calcorr_ref.(gasnames{1,i})=binref.(gasnames{1,i});     
        end

        % plot data before correction
        if plot_requested 
            plot_2point_cal_corr(data,bindata,binref,gasnames,true_gas1,true_values1,true_gas2,true_values2,target_idx,ind_gas1,ind_gas2,sssplot,'Before calibration correction');
        end
        
        % apply correction
        for j=1:width(target_idx)
            for i=1:width(gastypeflag)
                   xxx=min(binref.(gasnames{1,ind_gas1})(1,target_idx(1,j)),binref.(gasnames{1,ind_gas2})(1,target_idx(1,j)));
                   yyy=min(true_values1(1,j),true_values2(1,j));
                   bindata.(gasnames{1,i})(:,target_idx(1,j))=sss(j,1)*(bindata.(gasnames{1,i})(:,target_idx(1,j))-xxx)+yyy;     
            end
            bindata.all(:,target_idx(1,j))=sss(j,1)*(bindata.all(:,target_idx(1,j))-xxx)+yyy;
        end
        
        % recalculate ref conditions
        [ref, binref, bindata] = calc_ref_conditions(data,bindata,gasnames,gastypeflag);
        
        % recalculate linear fit coefficients true vs measured
        for i=1:width(target_idx)
            true=[true_values1(1,i) true_values2(1,i)];
            measured=[binref.(gasnames{1,ind_gas1})(1,target_idx(1,i)) binref.(gasnames{1,ind_gas2})(1,target_idx(1,i))];
            [sss(i,:),~]=polyfit(measured,true,1);
            [sssplot(i,:),~]=polyfit(true,measured,1);
        end
        
        % plot data after correction
        if plot_requested 
            plot_2point_cal_corr(data,bindata,binref,gasnames,true_gas1,true_values1,true_gas2,true_values2,target_idx,ind_gas1,ind_gas2,sssplot,'After calibration correction');
        end
    end
        
    time_elapsed=toc; fprintf('apply_cal_corr completed (execution time: %1.2f s)\n',time_elapsed); 

end

